package com.weface.task;

import cn.hutool.core.collection.CollUtil;
import com.weface.common.utils.RedisUtil;
import com.weface.component.MenuService;
import com.weface.entity.MenuTagsEntity;
import com.weface.entity.UserTagEntity;
import com.weface.entity.UserMenusEntity;
import com.weface.service.MenuTagsService;
import com.weface.service.UserTagService;
import com.weface.service.UserMenusService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Collectors;

/**
 * @author Administrator
 * @CreateUser: Administrator
 * @CreateTime: 2021/11/4
 */
@Component
@Slf4j
@EnableScheduling
public class UserTagsTask {

    @Value("${gexiang.user_tag_id}")
    private String userTagId;
    @Value("${gexiang.push_max_size}")
    private String pushMaxSize;

    @Autowired
    private MenuService menuService;
    @Autowired
    private UserTagService userTagService;
    @Autowired
    private UserMenusService userMenusService;
    @Autowired
    private MenuTagsService menuTagsService;

    @Autowired
    private ThreadPoolTaskExecutor asyncServiceExecutor;

    /**
     * 更新用户标签
     */
    @Scheduled(cron = "0 20 11 * * ?")
    public void updateUserTags() {
        try {
            //获取每次处理的id起始值
            String userId = RedisUtil.StringOps.get("user_tag_id");
            if (StringUtils.isBlank(userId)) {
                userId = userTagId;
            }
            int id = Integer.parseInt(userId);

            //获取每次处理最大数
            String pushSize = RedisUtil.StringOps.get("push_max_size");
            if (StringUtils.isBlank(pushSize)) {
                RedisUtil.StringOps.set("push_max_size", pushMaxSize);
                pushSize = pushMaxSize;
            }
            int max = Integer.parseInt(pushSize);

            //存储用户标签信息
            List<UserMenusEntity> userMenusList = new ArrayList<>();
            //获取标签列表
            List<MenuTagsEntity> tags = menuTagsService.list();



            //获取小于起始值,且更新时间为当前时间用户信息
            List<UserTagEntity> beforeUser = userTagService.findUserByTodayAndIdBefore(id);
            //过滤用户gid
            List<String> beforeGid = beforeUser.stream().map(UserTagEntity::getGid).distinct().collect(Collectors.toList());

            //获取更新总数量
            int size = beforeGid.size();
            //如果更新数量大于总量,截取每次处理总量的更新用户信息
            if (size > max) {
                beforeGid = beforeGid.subList(0, max - 1);
            } else {
                //最大处理数减去当天更新处理数,得到剩余处理数,获取起始值后的新增的用户信息
                List<UserTagEntity> afterUser = userTagService.findUserByIdAfter(Integer.parseInt(userId), max - size);
                //如果当天新增数不为空
                if (CollUtil.isNotEmpty(afterUser)) {
                    //获取用户最后一条信息
                    UserTagEntity afterUserInfo = afterUser.get(afterUser.size() - 1);
                    //并覆盖起始值
                    RedisUtil.StringOps.set("user_tag_id", String.valueOf(afterUserInfo.getId()));
                    //过滤用户gid
                    List<String> afterGid = afterUser.stream().map(UserTagEntity::getGid).distinct().collect(Collectors.toList());
                    //调用个像接口获取新增用户标签
                    List<UserMenusEntity> afterTag = getUserTag(afterGid, afterUser, tags);
                    if (CollUtil.isNotEmpty(afterTag)) {
                        userMenusList.addAll(afterTag);
                    }
                }
            }
            //调用个像接口获取更新用户标签
            List<UserMenusEntity> beforeTag = getUserTag(beforeGid, beforeUser, tags);
            if (CollUtil.isNotEmpty(beforeTag)) {
                userMenusList.addAll(beforeTag);
            }
            //批量插入用户标签信息
            userMenusService.batchInsert(userMenusList);
            log.error("执行结束{}", asyncServiceExecutor.getPoolSize());
        } catch (Exception e) {
            e.printStackTrace();
            log.error("执行定时更新标签任务失败!");
        }
    }

    /**
     * 获取个像用户标签信息
     *
     * @param gid      用户gid
     * @param userInfo 用户信息
     * @param tag      标签信息
     * @return 查询到的用户标签信息
     * @throws InterruptedException 网络异常
     */
    private List<UserMenusEntity> getUserTag(List<String> gid, List<UserTagEntity> userInfo, List<MenuTagsEntity> tag) throws InterruptedException {
        //存储用户标签信息
        List<UserMenusEntity> userTags = new ArrayList<>();
        //获取用户gid总量
        int size = gid.size();
        //每个线程处理1000调数据
        int limit = 200;
        //获取总页数
        int totalPage = (size % limit == 0) ? (size / limit) : (size / limit + 1);
        CountDownLatch latch = new CountDownLatch(totalPage);
        //创建线程
        for (int i = 0; i < totalPage; i++) {
            int finalI = i;
            Runnable runnable = () -> {
                //截取每次处理数据
                List<String> list;
                if (finalI == totalPage - 1) {
                    list = gid.subList(finalI * limit, size);
                } else {
                    list = gid.subList(finalI * limit, (finalI + 1) * limit);
                }
                if (CollectionUtils.isNotEmpty(list)) {
                    //获取个像数据
                    Map<String, Object> android = menuService.getUserTags(list, tag);
                    if (android != null) {
                        //解析个像数据
                        List<UserMenusEntity> tagUser = menuService.getTagUser(android, userInfo);
                        userTags.addAll(tagUser);
                    }
                }
                latch.countDown();
            };
            asyncServiceExecutor.execute(runnable);
        }
        latch.await();
        return userTags;
    }

}
