package com.weface.config;

import cn.hutool.core.io.IoUtil;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

/**
 * @author Aleyn
 * 1.重复获取HttpServletRequest中的inputStream 用以获取json请求参数
 * 2.请求参数为二进制数据，controller无法正常映射，需要重写参数
 * @since 2021/11/19 16:00
 */
public class OnceRequestWrapper extends HttpServletRequestWrapper {

    /**
     * json请求参数
     */
    private final String body;

    /**
     * Map参数
     */
    private final Map<String, String[]> params = new HashMap<>();

    public OnceRequestWrapper(HttpServletRequest request) throws IOException {
        super(request);
        this.body = getJsonBody(request);
        //将参数表，赋予给当前的Map以便于持有request中的参数
        this.params.putAll(request.getParameterMap());
    }

    /**
     * 重载构造方法
     */

    public OnceRequestWrapper(HttpServletRequest request, Map<String, Object> extendParams) throws IOException {
        this(request);
        //这里将扩展参数写入参数表
        addAllParameters(extendParams);
    }

    private String getJsonBody(HttpServletRequest request) throws IOException {
        String body;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IoUtil.copy(request.getInputStream(), baos);
        body = baos.toString();
        return body;
    }


    /**
     * @return ServletInputStream 重写getInputStream()
     */
    @Override
    public ServletInputStream getInputStream() {

        //每次调用此方法时将数据流中的数据读取出来，然后再回填到InputStream之中
        //解决通过@RequestBody和@RequestParam（POST方式）读取一次后控制器拿不到参数问题
        final ByteArrayInputStream bais = new ByteArrayInputStream(this.body.getBytes());
        return new ServletInputStream() {

            @Override
            public boolean isFinished() {
                return false;
            }

            @Override
            public boolean isReady() {
                return false;
            }

            @Override
            public void setReadListener(ReadListener listener) {

            }

            @Override
            public int read() {
                return bais.read();
            }
        };
    }

    @Override
    public BufferedReader getReader() {
        return new BufferedReader(new InputStreamReader(this.getInputStream()));
    }

    public String getBody() {
        return this.body;
    }


    /**
     * 在获取所有的参数名,必须重写此方法，否则对象中参数值映射不上
     */
    @Override
    public Enumeration<String> getParameterNames() {
        return new Vector(params.keySet()).elements();
    }

    /**
     * 重写getParameter方法
     *
     * @param name 参数名
     * @return 返回参数值
     */
    @Override
    public String getParameter(String name) {
        String[] values = params.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }

    @Override
    public String[] getParameterValues(String name) {
        String[] values = params.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values;
    }

    /**
     * 增加多个参数
     *
     * @param otherParams 增加的多个参数
     */
    public void addAllParameters(Map<String, Object> otherParams) {
        for (Map.Entry<String, Object> entry : otherParams.entrySet()) {
            addParameter(entry.getKey(), entry.getValue());
        }
    }

    /**
     * 增加参数
     *
     * @param name  参数名
     * @param value 参数值
     */
    public void addParameter(String name, Object value) {
        if (value != null) {
            if (value instanceof String[]) {
                params.put(name, (String[]) value);
            } else if (value instanceof String) {
                params.put(name, new String[]{(String) value});
            } else {
                params.put(name, new String[]{String.valueOf(value)});
            }
        }
    }
}
