package com.weface.serviceimpl;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.weface.common.utils.Constant;
import com.weface.common.utils.DES;
import com.weface.common.utils.RedisUtil;
import com.weface.common.utils.SnowIdUtil;
import com.weface.dao.PushLogDao;
import com.weface.dto.InformForm;
import com.weface.entity.NavigateConfigEntity;
import com.weface.entity.PushLogEntity;
import com.weface.service.PushLogService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Slf4j
@Transactional(rollbackFor = {Exception.class})
@Service("pushLogService")
public class PushLogServiceImpl extends ServiceImpl<PushLogDao, PushLogEntity> implements PushLogService {


    @Override
    public void saveLog(JSONObject data, InformForm informForm, String appId) {
        try {
            List<PushLogEntity> logList = new ArrayList<>();
            for (String key : data.keySet()) {
                Long id = SnowIdUtil.nextId();
                PushLogEntity pushLog = getPushLog(id, informForm, key, data.toString(), 1);
                logList.add(pushLog);
                RedisUtil.HashOps.hPutEx(Constant.PUSH_TASK_INFO, key, String.valueOf(id), 2L, TimeUnit.HOURS);
            }
            if (!CollectionUtils.isEmpty(logList)) {
                this.saveBatch(logList);
            }
        } catch (Exception e) {
            log.error("保存推送日志错误:{}", e.getMessage());
            e.printStackTrace();
        }
    }

    @Override
    public void saveMobLog(JSONObject data, InformForm informForm, boolean isBatch) {
        try {
            Long id = SnowIdUtil.nextId();

            String batchId;
            if (isBatch) {
                batchId = "toMultiPush";
            } else {
                batchId = data.getJSONObject("res").getString("batchId");
            }

            PushLogEntity pushLog = getPushLog(id, informForm, batchId, data.toString(), 2);
            this.save(pushLog);
        } catch (Exception e) {
            log.error("保存mob推送日志错误:{}", e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * 推送日志存储
     *
     * @param id         id
     * @param informForm 推送内容
     * @param taskId     任务编号
     * @param result     推送结果
     * @param pushType   推送类型 1:个推 2:袤博
     * @return 执行结果
     * @throws Exception 异常
     */
    private PushLogEntity getPushLog(Long id, InformForm informForm, String taskId, String result, Integer pushType) throws Exception {
        PushLogEntity pushLogEntity = new PushLogEntity();
        pushLogEntity.setId(id);
        pushLogEntity.setAppName(informForm.getEquipmentType());
        pushLogEntity.setPushType(pushType);
        if (StringUtils.isNotEmpty(taskId) && taskId.length() > 200) {
            taskId = taskId.substring(0, 200);
        }
        pushLogEntity.setTaskId(taskId);
        pushLogEntity.setResult(result);
        String phone = informForm.getPhone();
        if (StringUtils.isNotEmpty(phone)) {
            DES des = DES.getInstanceDes();
            pushLogEntity.setPhone(des.encrypt(phone));
        }
        Integer messageTemplate = informForm.getMessageTemplate();
        if (messageTemplate != null) {
            pushLogEntity.setMessageTemplate(messageTemplate);
        }
        String cid = informForm.getCid();
        String pushTarget;
        if (StringUtils.isEmpty(cid)) {
            pushTarget = "all";
        } else {
            int length = cid.length();
            if (length > 200) {
                pushTarget = "toListMessage";
            } else {
                pushTarget = cid;
            }
        }
        pushLogEntity.setPushTarget(pushTarget);
        String body = informForm.getBody();
        pushLogEntity.setPushContent(body);
        pushLogEntity.setArriveStatus(1001);
        String jsonStr = JSONUtil.toJsonStr(informForm);
        pushLogEntity.setBas1(jsonStr);
        pushLogEntity.setUpdateTime(new Date());
        return pushLogEntity;
    }

    /**
     * 查询日志
     *
     * @param taskId 任务id
     * @return 日志信息
     */
    @Override
    public PushLogEntity getPushLog(String taskId) {
        List<PushLogEntity> list = this.list(new LambdaQueryWrapper<PushLogEntity>()
                .eq(PushLogEntity::getTaskId, taskId));
        if (!CollectionUtils.isEmpty(list)) {
            return list.get(0);
        } else {
            return null;
        }
    }

    /**
     * 查询推送日志
     *
     * @param time 时间
     * @return 推送日志
     */
    @Override
    public List<PushLogEntity> getPushLogByTime(Integer time) {
        return this.baseMapper.getPushLogByTime(time);
    }

    /**
     * 获取已缴费用户id
     *
     * @return 用户id列表
     */
    @Override
    public List<String> getOrderUidList() {
        return this.baseMapper.getOrderUidList();
    }

    /**
     * 查询当月1-14号交费记录的用户id
     *
     * @return 未缴费用户id
     */
    @Override
    public List<String> getPayCostRecord() {
        return this.baseMapper.getPayCostRecord();
    }

    /**
     * 根据配置名称获取配置信息 并缓存10分钟
     *
     * @param configName 配置名称
     * @return 配置信息
     */
    @Override
    public NavigateConfigEntity getConfigByName(String configName) {
        NavigateConfigEntity config;
        String key = "population:navigateConfig:" + configName;
        String configBody = RedisUtil.StringOps.get(key);
        if (StringUtils.isEmpty(configBody)) {
            List<NavigateConfigEntity> list = this.baseMapper.getConfigByName(configName);
            if (CollectionUtils.isEmpty(list)) {
                return null;
            }
            config = list.get(0);
            RedisUtil.StringOps.setEx(key, JSONObject.toJSONString(config), 10, TimeUnit.MINUTES);
        } else {
            config = JSONObject.parseObject(configBody, NavigateConfigEntity.class);
        }
        return config;
    }
}