package com.weface.component;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.lang.TypeReference;
import cn.hutool.core.util.IdUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.http.HtmlUtil;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.weface.code.CommonResult;
import com.weface.common.utils.CommonUtil;
import com.weface.common.utils.Constant;
import com.weface.common.utils.Model;
import com.weface.dto.InformForm;
import com.weface.dto.MultiInformForm;
import com.weface.dto.MultiMsgDTO;
import com.weface.service.PushLogService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.lang.reflect.Type;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : Administrator
 * @date : 2022/5/30 17:49
 * mob 推送
 */
@Component
@Slf4j
public class MobPushService {
    /**
     * mob推送key
     */
    private static final String APP_KEY = "3511c2579cad6";
    /**
     * mob推送密钥
     */
    private static final String APP_SECRET = "f446a56b98aae09beb5721ebdbe0d2e8";

    @Autowired
    private PushLogService pushLogService;

    /**
     * 批量推送最大消息体数量
     */
    private static final int MAX_ITEM_SIZE = 20;

    /**
     * 批量推送的消息体内最大推送设备数
     */
    private static final int MAX_ALIAS = 200;
    /**
     * mob 推送
     *
     * @param informForm 推送信息
     * @param target     推送目标类型
     * @return 执行结果
     */
    public CommonResult mobPush(InformForm informForm, Constant.PushTarget target) {
        String pushUrl = "http://api.push.mob.com/v3/push/createPush";
        Map<String, Object> request = new HashMap<>();
        //枚举值 webapi
        request.put("source", "webapi");
        //自定义任务id（用户自自定义生成且唯一、不能重复）
        request.put("workno", IdUtil.fastUUID());
        //appkey
        request.put("appkey", APP_KEY);
        informForm.setUrl(HtmlUtil.unescape(informForm.getUrl()));
        String[] split = new String[0];
        if (target.getKey() != 1) {
            //推送目标
            String cid = informForm.getCid();
            if (cid.contains(",")) {
                split = cid.split(",");
            } else {
                split = new String[]{cid};
            }
        }
        Map<String, Object> pushTarget = getPushTarget(target.getKey(), split);
        request.put("pushTarget", pushTarget);

        //推送展示细节配置
        Map<String, Object> pushNotify = getPushNotify(informForm);
        request.put("pushNotify", pushNotify);
        Map<String, Object> pushForward = getPushForward();
        request.put("pushForward", pushForward);
        //厂商特殊配置
        Map<String, Object> pushFactoryExtra = getPushFactoryExtra();
        request.put("pushFactoryExtra", pushFactoryExtra);
        //groupId: AB分组测试ID
        request.put("groupId", target.getValue());
        String body = generalPost(request, pushUrl);
        if (body != null) {
            pushLogService.saveMobLog(JSONObject.parseObject(body), informForm, false);
            return CommonResult.success();
        }
        return null;
    }

    /**
     * 厂商特殊渠道配置
     *
     * @return map
     */
    private Map<String, Object> getPushFactoryExtra() {
        Map<String, Object> pushFactoryExtra = new HashMap<>();
        Map<String, Object> huaweiExtra = new HashMap<>();
        huaweiExtra.put("importance", "NORMAL");
        pushFactoryExtra.put("huaweiExtra", huaweiExtra);
        Map<String, Object> vivoExtra = new HashMap<>();
        vivoExtra.put("classification", "1");
        pushFactoryExtra.put("vivoExtra", vivoExtra);
        return pushFactoryExtra;
    }

    /**
     * mob别名批量推送
     * 与袤博技术人员确认：建议一个推送任务设置20 items，一个items可以设置200个推送设备
     * @param multiInformForm 推送内容
     * @return 执行结果
     */
    public CommonResult batchMobPush(MultiInformForm multiInformForm) {
        // 消息体数量
        List<MultiMsgDTO> itemList = multiInformForm.getMsgList();
        List<MultiMsgDTO> splitItemList = new ArrayList<MultiMsgDTO>();

        // 判断每个消息体中的推送设备数是否达到最大限制，
        // 如果达到最大限制，则从中分割出N个小于最大限制的消息体，并将其添加到原始的消息体数量中
        for (MultiMsgDTO msg : itemList) {
            if (msg.getAlias().length > MAX_ALIAS) {
                // 从当前的消息体中拆分出N个小于最大限制的消息体
                List<MultiMsgDTO> subItemList = splitExceedLimitMsg(msg);
                // 将新的消息体集合统一放到的新的消息体集合中
                splitItemList.addAll(subItemList);
            } else {
                msg.setItemId(IdUtil.fastUUID());
            }
        }

        if (!CollectionUtils.isEmpty(splitItemList)) {
            // 将所有划分后的消息体放到原始的消息体集合中
            itemList.addAll(splitItemList);
        }

        // 删除没有itemId的元素（超过最大限制的元素）
        itemList.removeIf(x -> StringUtils.isBlank(x.getItemId()));

        // 判断消息体的数量是否达到上限
        // 如果达到最大限制，则将其划分成N个小于最大限制的推送任务
        if (itemList.size() > MAX_ITEM_SIZE) {
            List<MultiInformForm> informForms = splitExceedLimitForm(multiInformForm);
            batchMobPushExecute(informForms);
        } else {
            // 直接执行当前推送任务
            batchMobPushExecute(Collections.singletonList(multiInformForm));
        }

        return CommonResult.success();
    }

    /**
     * 执行批量推送
     * @param multiInformFormList 推送任务集合
     */
    public void batchMobPushExecute(List<MultiInformForm> multiInformFormList) {
        String pushUrl = "http://api.push.mob.com/v3/push/createMulti";

        for (MultiInformForm form : multiInformFormList) {
            Map<String, Object> request = new HashMap<>(2);

            Map<String, Object> pushWork = new HashMap<>();
            //枚举值 webapi
            pushWork.put("source", "webapi");
            //Mob-appkey
            pushWork.put("appkey", APP_KEY);
            //推送展示细节配置
            InformForm informForm = new InformForm();
            BeanUtils.copyProperties(form, informForm);
            Map<String, Object> pushNotify = getBatchPushNotify(informForm);

            pushWork.put("pushNotify", pushNotify);
            //link 相关打开配置
            Map<String, Object> pushForward = getPushForward();
            pushWork.put("pushForward", pushForward);
            //groupId: AB分组测试ID
            pushWork.put("groupId", Constant.PushTarget.ALIAS.getKey());
            request.put("pushWork", pushWork);

            MultiMsgDTO[] items = form.getMsgList().toArray(new MultiMsgDTO[form.getMsgList().size()]);
            request.put("items", items);
            String body = generalPost(request, pushUrl);
            log.info("批量推送结果:{}", body);

            if (body != null) {
                batchMobPushSaveLog(form, body);
            }
        }
    }

    public void batchMobPushSaveLog(MultiInformForm multiInformForm, String resultBody) {
        for (MultiMsgDTO msg : multiInformForm.getMsgList()) {
            InformForm informForm = new InformForm();
            BeanUtils.copyProperties(multiInformForm, informForm);
            informForm.setBody(msg.getContent());
            String cid = String.join(",", msg.getAlias());
            informForm.setCid(cid);
            informForm.setMessageTemplate(3);

            pushLogService.saveMobLog(JSONObject.parseObject(resultBody), informForm, true);
        }
    }


    /**
     * 将超出 最大推送设备数的消息体进行分割
     * @param msg 消息体
     * @return 分割后的消息体的集合
     */
    private List<MultiMsgDTO> splitExceedLimitMsg(MultiMsgDTO msg) {
        // 计算最终从拆分的消息体的数量
        Integer itemCount = CommonUtil.getGroupCount(msg.getAlias().length, MAX_ALIAS);

        // 定义返回的消息体集合
        List<MultiMsgDTO> itemList = new ArrayList<MultiMsgDTO>(itemCount);

        // 取出所有原消息体中的推送设备
        List<String> alias = new ArrayList<>(Arrays.asList(msg.getAlias()));

        // 将推送设备进行拆分
        for (int i = 0; i < itemCount; i++) {
            // 当前子消息体内的推送设备列表
            List<String> subAlias = new ArrayList<>();
            if (i == itemCount - 1) {
                subAlias = alias.subList(MAX_ALIAS * i, alias.size() - 1);
                subAlias.add(alias.get(alias.size() - 1));
            } else {
                subAlias = alias.subList(MAX_ALIAS * i, (MAX_ALIAS * i) + MAX_ALIAS);
            }

            MultiMsgDTO item = new MultiMsgDTO();
            BeanUtils.copyProperties(msg, item);
            item.setAlias(subAlias.toArray(new String[subAlias.size()]));
            item.setItemId(IdUtil.fastUUID());

            itemList.add(item);
        }

        return itemList;
    }

    /**
     * 将超出 最大推送消息体数量的任务进行分割
     * @param form 推送任务
     * @return 分割后的推送任务的集合
     */
    private List<MultiInformForm> splitExceedLimitForm(MultiInformForm form) {
        // 计算最终从拆分的推送任务的数量
        Integer formCount = CommonUtil.getGroupCount(form.getMsgList().size(), MAX_ITEM_SIZE);

        // 定义返回的推送任务集合
        List<MultiInformForm> subFormList = new ArrayList<MultiInformForm>(formCount);

        // 取出所有原推送任务中的消息体
        List<MultiMsgDTO> msgList = form.getMsgList();

        // 将消息体进行拆分
        for (int i = 0; i < formCount; i++) {
            MultiInformForm subForm = new MultiInformForm();
            subForm.setPushType(form.getPushType());
            subForm.setEquipmentType(form.getEquipmentType());
            subForm.setUrl(form.getUrl());

            // 当前子推送任务内的消息体
//            List<MultiMsgDTO> subMsgList = new ArrayList<>();
            if (i == formCount - 1) {
                List<MultiMsgDTO> subMsgList = msgList.subList(MAX_ITEM_SIZE * i, msgList.size() - 1);
                // 不再对原集合操作，避免出现java.util.ConcurrentModificationException异常
                if (subMsgList.size() == 0) {
                    subForm.setMsgList(new ArrayList<>(Collections.singletonList(msgList.get(msgList.size() - 1))));
                } else {
                    ArrayList<MultiMsgDTO> enableOperateList = new ArrayList<>(subMsgList);
                    enableOperateList.add(msgList.get(msgList.size() - 1));
                    subForm.setMsgList(enableOperateList);
                }
            } else {
                List<MultiMsgDTO> subMsgList = msgList.subList(MAX_ITEM_SIZE * i, (MAX_ITEM_SIZE * i) + MAX_ITEM_SIZE);
                subForm.setMsgList(subMsgList);
            }

            subFormList.add(subForm);
        }

        return subFormList;
    }

    /**
     * 通过 workId查询当前消息推送结果
     *
     * @param workId 推送id
     * @return 是否收到
     */
    public boolean getPushInfoById(String workId) {
        String reqUrl = "http://api.push.mob.com/v3/stats/getByWorkId";
        Map<String, Object> request = new HashMap<>();
        //appkey
        request.put("appkey", APP_KEY);
        request.put("workId", workId);
        String body = generalPost(request, reqUrl);
        log.error("查询回调结果:{}", body);
        if (JSONUtil.isJson(body)) {
            JSONObject json = JSONObject.parseObject(body);
            if (json.containsKey("status") && json.getInteger("status").equals(200)) {
                if (json.containsKey("res")) {
                    JSONObject res = json.getJSONObject("res");
                    if (Objects.nonNull(res)) {
                        if (res.containsKey("android")) {
                            JSONObject android = res.getJSONObject("android");
                            if (Objects.nonNull(android)) {
                                if (android.containsKey("deliverFailNum")) {
                                    Integer deliverFailNum = android.getInteger("deliverFailNum");
                                    return deliverFailNum <= 0;
                                }
                            }
                        } else if (res.containsKey("ios")) {
                            JSONObject ios = res.getJSONObject("ios");
                            if (Objects.nonNull(ios)) {
                                if (ios.containsKey("deliverFailNum")) {
                                    Integer deliverFailNum = ios.getInteger("deliverFailNum");
                                    return deliverFailNum <= 0;
                                }
                            }
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * 批量推送
     *
     * @param informForm 推送信息
     * @return 执行结果
     */
    private Map<String, Object> getBatchPushNotify(InformForm informForm) {
        //推送展示细节配置
        Map<String, Object> pushNotify = new HashMap<>(8);
        //1 android;2 ios
        pushNotify.put("plats", new Integer[]{1, 2});
        //推送类型：1通知；2自定义
        pushNotify.put("type", 1);
        //android通知消息, type=1, android
        Map<String, Object> androidNotify = getAndroidNotify(informForm);
        if (!androidNotify.isEmpty()) {
            pushNotify.put("androidNotify", androidNotify);
        }
        //ios通知消息, type=1, ios
        Map<String, Object> iosNotify = getIosNotify();
        if (!iosNotify.isEmpty()) {
            pushNotify.put("iosNotify", iosNotify);
        }
        if (informForm.getSpeed() != null) {
            pushNotify.put("speed", informForm.getSpeed());
        }
        //推送时，不走厂商通道。1: 开启，不走厂商通道；其它，关闭 ，默认关闭。
        pushNotify.put("skipFactory", 0);
        //推送策略： * 1:先走tcp，再走厂商 * 2:先走厂商，再走tcp * 3:只走厂商 * 4:只走tcp (厂商透传policy只支持策略3或4)
        pushNotify.put("policy", 1);
        //拓展参数
        pushNotify.put("extrasMapList", getExtrasMapList(informForm));
        return pushNotify;
    }

    /**
     * 获取推送目标
     *
     * @param type   推送类型
     * @param target 推送目标
     * @return 推送目标配置
     */
    private Map<String, Object> getPushTarget(Integer type, Object target) {
        //推送目标
        Map<String, Object> pushTarget = new HashMap<>(2);
        //目标类型：1广播；2别名；3标签；4regid；5地理位置；6用户分群；9复杂地理位置推送；14fileid推送
        pushTarget.put("target", type);
        //集合元素限制1000个以内
        if (target instanceof String[]) {
            String[] str = (String[]) target;
            if (str.length > 1000) {
                return null;
            }
        }
        //推送目标
        switch (type) {
            case 1:
                //target:1 => 1广播
                break;
            case 2:
                //arget:2 => 设置推送别名集合["alias1","alias2"]集合元素限制1000个以内
                pushTarget.put("alias", target);
                break;
            case 3:
                //target:3 => 设置推送标签集合["tag1","tag2"]集合元素限制1000个以内
                pushTarget.put("tags", target);
                //target:3 => 标签组合方式：1并集；2交集；3补集(3暂不考虑)
                pushTarget.put("tagsType", 1);
                break;
            case 4:
                //target:4 => 设置推送Registration Id集合["id1","id2"]集合元素限制1000个以内
                pushTarget.put("rids", target);
                break;
            case 5:
                //target:5 => 推送地理位置 城市，地理位置推送时（示例：上海市注意要带“市”），city, province, country 必须有一个不为空
                /*if (str.length > 0) {
                    pushTarget.put("province", str[0]);
                }
                if (str.length > 1) {
                    pushTarget.put("city", str[1]);
                }
                if (str.length > 2) {
                    pushTarget.put("country", str[2]);
                }*/
                if (target instanceof Map) {
                    Map<String, String> address = Convert.convert(new TypeReference<Map<String, String>>() {
                        @Override
                        public Type getType() {
                            return super.getType();
                        }
                    }, target);
                    pushTarget.putAll(address);
                }
                break;
            case 6:
                pushTarget.put("block", target);
                break;
            case 9:
                pushTarget.put("pushAreas", target);
                break;
            case 14:
                pushTarget.put("fileId", target);
                break;
            default:
                break;
        }
        return pushTarget;
    }

    /**
     * 推送展示细节配置
     *
     * @param informForm 推送内容细节
     * @return 推送展示配置
     */
    private Map<String, Object> getPushNotify(InformForm informForm) {
        Map<String, Object> pushNotify = getBatchPushNotify(informForm);
        //推送内容, 如果内容长度超过厂商的限制, 则内容会被截断. vivo不允许纯表情
        pushNotify.put("content", informForm.getBody());
        //如果不设置，则默认的通知标题为应用的名称。如果标题长度超过厂商的限制, 则标题会被截断. vivo不允许纯表情
        pushNotify.put("title", informForm.getTitle());
        return pushNotify;
    }

    /**
     * link 相关打开配置 目前默认打开首页
     *
     * @return 消息跳转地址配置
     */
    private Map<String, Object> getPushForward() {
        Map<String, Object> pushForward = new HashMap<>(1);
        pushForward.put("nextType", 0);
        return pushForward;
    }

    /**
     * 链接跳转地址配置
     *
     * @param informForm 推送信息
     * @return 跳转地址配置
     */
    private List<Map<String, Object>> getExtrasMapList(InformForm informForm) {
//        String pack = "com.weface.kksocialsecurity";
//        String component = pack + "/com.weface.kksocialsecurity.activity.WellcomeActivity";
        Integer pushType = informForm.getPushType();
        int push;
        if (pushType == null || pushType == 0) {
            push = 0;
        } else if (pushType == 1) {
            push = 2;
        } else {
            push = pushType;
        }
        List<Map<String, Object>> list = new ArrayList<>(2);
        //设置包名
//        list.add(getFiled("package", pack));
        //设置路径
//        list.add(getFiled("component", component));
        //推送内容
//        list.add(getFiled("content", informForm.getBody()));
        //推送类型 0 h5 2 原生
        list.add(getFiled("push_type", push));
        //推送地址
        list.add(getFiled("url", informForm.getUrl()));
        //推送标题
//        list.add(getFiled("title", informForm.getTitle()));
        return list;
    }

    /**
     * 获取字段
     *
     * @param key   key
     * @param value value
     * @return 字段
     */
    private Map<String, Object> getFiled(String key, Object value) {
        Map<String, Object> map = new HashMap<>(2);
        map.put("key", key);
        map.put("value", value);
        return map;
    }

    /**
     * 安卓通知细节配置
     *
     * @param informForm 推送消息内容
     * @return 安卓细节
     */
    private Map<String, Object> getAndroidNotify(InformForm informForm) {
        Map<String, Object> androidNotify = new HashMap<>(1);
        androidNotify.put("warn", "123");
        Integer style = informForm.getStyle();
        if (Objects.isNull(style)) {
            return androidNotify;
        }
        androidNotify.put("style", style);
        String content = informForm.getContent();
        if (style == 1 || style == 2) {
            androidNotify.put("content", new String[]{content});
        }
        String image = informForm.getImage();
        if (StringUtils.isNotEmpty(image)) {
            androidNotify.put("image", image);
        }
        return androidNotify;
    }

    /**
     * customStyle：安卓通知自定义样式
     *
     * @return 安卓细节
     */
    private Map<String, Object> getCustomStyle() {
        Map<String, Object> map = new HashMap<>();
        map.put("styleNo", 1);
        map.put("backgroundUrl", "https://socialsecurity.oss-cn-beijing.aliyuncs.com/userPhoto/ycfj.png");
        map.put("smallIcons", "http://i.weface.com.cn/banners/tcformphoto/certificationOldSuucess01.gif");
        map.put("buttonCopy", "测试文案");
        map.put("buttonJumpUrl", "https://www.hutool.cn/");
        return map;
    }

    /**
     * ios通知细节配置
     *
     * @return ios细节
     */
    private Map<String, Object> getIosNotify() {
        Map<String, Object> iosNotify = new HashMap<>(0);
        return iosNotify;
    }


    /**
     * 推送回调配置
     *
     * @param url    回调地址
     * @param params 回调参数
     * @return 回调配置
     */
    private Map<String, Object> getPushCallback(String url, Object params) {
        Map<String, Object> pushCallback = new HashMap<>(2);
        pushCallback.put("url", url);
        pushCallback.put("params", params);
        return pushCallback;
    }

    /**
     * 自定义参数配置
     *
     * @param title   自定义标题
     * @param content 自定义内容
     * @return 自定义参数配置
     */
    private Map<String, Object> getCustomNotify(String title, String content) {
        Map<String, Object> customNotify = new HashMap<>(2);
        customNotify.put("customTitle", title);
        customNotify.put("customType", content);
        return customNotify;
    }

    /**
     * mob 通用post请求
     *
     * @param map 请求参数
     * @param url 请求地址
     * @return 执行结果
     */
    private String generalPost(Map<String, Object> map, String url) {
        String request;
        if (map == null) {
            request = "";
        } else {
            request = JSONObject.toJSONString(map);
        }
        log.info("mob推送内容:{}", request);
        HttpRequest post = HttpUtil.createPost(url);
        post.header("Content-Type", "application/json");
        post.header("key", APP_KEY);
        post.header("sign", getSign(request));
        String body = post.body(request).execute().body();
        log.info("mob推送结果:{}", body);
        if (JSONUtil.isJson(body)) {
            JSONObject json = JSONObject.parseObject(body);
            if (json.containsKey("status") && json.getInteger("status").equals(200)) {
                JSONObject res = (JSONObject) JSONObject.parseObject(body).get("res");
                if (!res.containsKey("errors")) {
                    return body;
                }
                Object errors = res.get("errors");
                if (ObjectUtils.isEmpty(errors)) {
                    return body;
                }

            }
        }
        return null;
    }

    /**
     * mob 通用get请求
     *
     * @param url 请求路径
     * @return 执行结果
     */
    private String generalGet(String url) {
        HttpRequest get = HttpUtil.createGet(url);
        get.header("key", APP_KEY);
        get.header("sign", getSign(""));
        String body = get.execute().body();
        log.info("推送结果:{}", body);
        if (JSONUtil.isJson(body)) {
            JSONObject json = JSONObject.parseObject(body);
            if (json.containsKey("status") && json.getInteger("status").equals(200))
                return body;
        }
        return null;
    }

    /**
     * 获取签名 请求体+AppSecret来进行MD5加密
     *
     * @param body 请求体
     * @return sign
     */
    private String getSign(String body) {
        return SecureUtil.md5(body + APP_SECRET);
    }
}
