package com.weface.component;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HtmlUtil;
import cn.hutool.http.HttpException;
import cn.hutool.http.HttpRequest;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.weface.code.CommonResult;
import com.weface.common.exception.RRException;
import com.weface.common.utils.Base64Util;
import com.weface.common.utils.HttpUtil;
import com.weface.common.utils.RedisUtil;
import com.weface.config.GeTuiApp;
import com.weface.config.GeTuiConfig;
import com.weface.dto.InformForm;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author Administrator
 * @CreateTime: 2021/11/2
 */
@Component
@Slf4j
public class GeTuiService {

    @Resource
    private GeTuiApp geTuiApp;
    public static final String GE_TUI_BASE_URL = "https://restapi.getui.com/v2/";
    public static final String BASEURL = "https://openapi-gi.getui.com/v2/";
    public static final String ENCODING = "UTF-8";
    public static final String CONTENT_TYPE = "application/json;charset=utf-8";

    /**
     * 设置token过期时间为12小时
     */
    private final static long TIMEOUT = (12 * 60 * 60);

    /**
     * 获取token并缓存一份
     *
     * @param key 缓存key
     * @return 从缓存中拿token
     */
    public String getAuthToken(String key) {
        GeTuiConfig kksh = getConfig("kksh");
        GeTuiConfig kksb = getConfig("kksb");
        GeTuiConfig kkwj = getConfig("kkwj");
        GeTuiConfig kkmz = getConfig("kkmz");
        String authToken = "";
        try {
            boolean flag = false;
            while (!flag) {
                flag = RedisUtil.LockOps.getLock("token", Thread.currentThread().getName());
                if (!flag) {
                    Thread.sleep(500L);
                }
            }
            //获取缓存内的token
            authToken = RedisUtil.StringOps.get(key);
            //如果不存在则调用个像接口获取
            if (StringUtils.isBlank(authToken)) {
                String geTui = null;
                switch (key) {
                    case "kk_sh_token":
                        authToken = getAuth(kksh.getAppId(), kksh.getAppkey(), kksh.getMastersecret());
                        if (ObjectUtil.isNull(authToken)) {
                            throw new RRException("获取看看生活个像token失败");
                        }
                        break;
                    case "kk_sb_token":
                        authToken = getAuth(kksb.getAppId(), kksb.getAppkey(), kksb.getMastersecret());
                        if (ObjectUtil.isNull(authToken)) {
                            throw new RRException("获取看看社保个像token失败");
                        }
                        break;
                    case "kk_sh_token_ge_tui":
                        authToken = getGeTuiAuth(kksh.getAppId(), kksh.getAppkey(), kksh.getMastersecret());
                        if (ObjectUtil.isNull(authToken)) {
                            throw new RRException("获取看看生活个推token失败");
                        }
                        geTui = key;
                        break;
                    case "kk_sb_token_ge_tui":
                        authToken = getGeTuiAuth(kksb.getAppId(), kksb.getAppkey(), kksb.getMastersecret());
                        if (ObjectUtil.isNull(authToken)) {
                            throw new RRException("获取看看社保个推token失败");
                        }
                        geTui = key;
                        break;
                    case "kk_wj_token_ge_tui":
                        authToken = getGeTuiAuth(kkwj.getAppId(), kkwj.getAppkey(), kkwj.getMastersecret());
                        if (ObjectUtil.isNull(authToken)) {
                            throw new RRException("获取看看卫建个推token失败");
                        }
                        geTui = key;
                        break;
                    case "kk_mz_token_ge_tui":
                        authToken = getGeTuiAuth(kkmz.getAppId(), kkmz.getAppkey(), kkmz.getMastersecret());
                        if (ObjectUtil.isNull(authToken)) {
                            throw new RRException("获取看看民政个推token失败");
                        }
                        geTui = key;
                        break;
                    default:
                        break;
                }
                if (geTui != null) {
                    //获取token过期时间并转为数值,使token提前5秒过期
                    RedisUtil.StringOps.setEx(key, authToken, (TIMEOUT * 2) - 5, TimeUnit.SECONDS);
                } else {
                    //获取成功后放入缓存并设置token过期时间
                    RedisUtil.StringOps.setEx(key, authToken, TIMEOUT, TimeUnit.SECONDS);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("加锁失败!");
        } finally {
            RedisUtil.LockOps.releaseLock("token", Thread.currentThread().getName());
        }
        return authToken;
    }

    /**
     * 获取个像token
     *
     * @return JSONObject
     */
    private String getAuth(String appId, String appKey, String masterSecret) {
        try {
            String getAccessTokenUrl = BASEURL + appId + "/auth_sign";
            Long timestamp = System.currentTimeMillis();
            JSONObject json = new JSONObject();
            json.put("sign", DigestUtils.sha256Hex(String.format("%s%d%s", appKey, timestamp, masterSecret)));
            json.put("timestamp", timestamp);
            HttpRequest post = HttpRequest.post(getAccessTokenUrl);
            String result = post.body(json.toString()).execute().body();
            JSONObject jsonObject = JSONObject.parseObject(result);
            if (jsonObject.containsKey("result") && "true".equals(jsonObject.getString("result"))) {
                String authtoken = jsonObject.getString("authtoken");
                log.info("获取个像token成功{}", authtoken);
                return authtoken;
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取个像token失败!");
        }
        return null;
    }

    /**
     * 获取个推token
     *
     * @return JSONObject
     */
    public String getGeTuiAuth(String appId, String appKey, String masterSecret) {
        try {
            // 获取token地址
            log.info("开始获取个推token");
            String getAccessTokenUrl = GE_TUI_BASE_URL + appId + "/auth";
            Long timestamp = System.currentTimeMillis();
            Map<String, Object> map = new HashMap<>();
            map.put("sign", Base64Util.getSign(appKey, timestamp, masterSecret));
            map.put("timestamp", timestamp);
            map.put("appkey", appKey);
            String result = HttpUtil.postGeneralUrl(getAccessTokenUrl, CONTENT_TYPE, JSONObject.toJSONString(map), ENCODING);
            if (HttpUtil.isJson2(result)) {
                JSONObject jsonObject = JSONObject.parseObject(result);
                if (jsonObject.containsKey("code") && jsonObject.getInteger("code") == 0) {
                    return jsonObject.getJSONObject("data").getString("token");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取个推token失败!");
        }
        return null;
    }

    /**
     * 获取appId
     *
     * @param appName app名称
     * @return appId
     */
    public String getAppId(String appName) {
        return getConfig(appName).getAppId();
    }

    /**
     * 获取app配置
     *
     * @param appName app名称
     * @return app配置
     */
    public GeTuiConfig getConfig(String appName) {
        return geTuiApp.getApps().get(appName);
    }

    /**
     * 获取看看生活个像
     *
     * @param gidList 用户gid
     * @param token   token
     * @return 请求结果
     * @throws Exception 抛出异常
     */
    public String queryTagKKSH(List<String> gidList, String token) throws Exception {
        return queryTag(gidList, token, getAppId("kksh"));
    }

    /**
     * 获取看看生活个像
     *
     * @param gidList 用户gid
     * @param token   token
     * @return 请求结果
     * @throws Exception 抛出异常
     */
    public String queryTagKKSB(List<String> gidList, String token) throws Exception {
        return queryTag(gidList, token, getAppId("kksb"));
    }

    /**
     * 根据giUid查询用户相关标签code
     *
     * @param gidList 用户gid
     * @param token   鉴权
     * @return 执行结果
     * @throws Exception 抛出异常
     */
    private String queryTag(List<String> gidList, String token, String appId) throws Exception {
        //请求url
        String getUsersCidUrl = "https://openapi-gi.getui.com/v3/" + appId + "/query_tag";
        //将token以及用户ID封装调用画像查询接口
        Map<String, Object> map = new HashMap<>(2);
        map.put("userIdList", gidList);
        map.put("token", token);
        //发送post请求拿到当前用户的所有图像
        return HttpUtil.postGeneralUrl(getUsersCidUrl, CONTENT_TYPE, JSONObject.toJSONString(map), ENCODING);
    }


    /**
     * 删除鉴权token
     *
     * @param token 接口调用凭证token
     * @param appId appId
     * @return 是否成功
     */
    public boolean delToken(String token, String appId) {
        try {
            String body = HttpRequest.delete(GE_TUI_BASE_URL + appId + "/auth" + token).execute().body();
            if (HttpUtil.isJson2(body)) {
                JSONObject jsonObject = JSONObject.parseObject(body);
                return jsonObject.containsKey("code") && jsonObject.getInteger("code") == 0;
            }
        } catch (HttpException e) {
            e.printStackTrace();
            log.error("删除个推鉴权token失败{}", e.getMessage());
        }
        return false;
    }

    /**
     * 【toSingle】执行别名单推
     *
     * @param messageTemplate 应用id
     * @param appId           消息模板
     * @param token           接口访问凭据
     * @return 消息推送结果
     */
    public String singlePushAlias(Integer device, MessageTemplate messageTemplate, String appId, String token) throws UnsupportedEncodingException {
        Map<String, Object> map = convertBeanToMap(device, false, 1, messageTemplate);
        //请求url
        String url = GE_TUI_BASE_URL + appId + "/push/single/alias";
        log.info("执行别名单推");
        return generalPost(url, token, map);
    }

    /**
     * 【toSingle】执行cid单推
     *
     * @param device          设备类型
     * @param messageTemplate 消息
     * @param appId           appId
     * @param token           鉴权
     * @return 执行结果
     */
    public String singlePushCid(Integer device, MessageTemplate messageTemplate, String appId, String token) throws UnsupportedEncodingException {
        Map<String, Object> map = convertBeanToMap(device, false, 0, messageTemplate);
        //请求url
        String url = GE_TUI_BASE_URL + appId + "/push/single/cid";
        log.info("执行cid单推");
        return generalPost(url, token, map);
    }

    /**
     * 创建消息拿到任务id
     *
     * @param device          设备类型
     * @param messageTemplate 消息模板
     * @param appId           appId
     * @param token           token
     * @return 任务id
     */
    public String getTaskId(Integer device, MessageTemplate messageTemplate, String appId, String token) throws UnsupportedEncodingException {
        Map<String, Object> map = convertBeanToMap(device, true, 1, messageTemplate);
        //请求url
        String url = GE_TUI_BASE_URL + appId + "/push/list/message";
        String body = generalPost(url, token, map);
        JSONObject jsonObject = JSONObject.parseObject(body);
        return (jsonObject.containsKey("code") && jsonObject.getInteger("code") == 0) ? jsonObject.getJSONObject("data").getString("taskid") : body;
    }

    /**
     * 【toList】执行cid批量推
     *
     * @param cid      cid数组
     * @param token    鉴权
     * @param taskId   任务id
     * @param is_async 是否异步
     * @return 执行结果
     */
    public String listPushCid(String[] cid, String token, String taskId, boolean is_async) {
        String url = GE_TUI_BASE_URL + getAppId("kksh") + "/push/list/cid";
        Map<String, Object> cids = new HashMap<>(1);
        cids.put("cid", cid);
        return listPushTaskId(url, token, taskId, is_async, cids);
    }

    /**
     * 【toList】执行别名批量推
     *
     * @param alias    别名
     * @param token    鉴权
     * @param taskId   任务id
     * @param is_async 是否异步
     * @return 执行结果
     */
    public String listPushAlias(String[] alias, String token, String taskId, boolean is_async) {
        String url = GE_TUI_BASE_URL + getAppId("kksh") + "/push/list/alias";
        Map<String, Object> aliass = new HashMap<>(1);
        aliass.put("alias", alias);
        return listPushTaskId(url, token, taskId, is_async, aliass);
    }

    /**
     * 批量推送通用
     *
     * @param url      请求路径
     * @param token    鉴权
     * @param taskId   任务id
     * @param is_async 是否异步
     * @param hashMao  推送目标用户
     * @return 执行结果
     */
    private String listPushTaskId(String url, String token, String taskId, boolean is_async, Map<String, Object> hashMao) {
        Map<String, Object> map = new HashMap<>(3);
        map.put("audience", hashMao);
        map.put("taskid", taskId);
        map.put("is_async", is_async);
        return generalPost(url, token, map);
    }

    /**
     * 群推
     *
     * @param informForm 消息
     * @return 执行状态
     */
    public CommonResult listPush(InformForm informForm) {
        MessageTemplate messageTemplate = new MessageTemplate();
        try {
            GeTuiConfig kksh = getConfig("kksh");
            GeTuiConfig kksb = getConfig("kksb");
            GeTuiConfig kkwj = getConfig("kkwj");
            GeTuiConfig kkmz = getConfig("kkmz");

            Integer device = informForm.getDevice();
            if (device == null) {
                device = 4;
            }
            Integer speed = informForm.getSpeed();
            messageTemplate.setSettings(new MessageTemplate.Settings(3600000, speed == null ? 0 : speed));

            MessageTemplate.PushMessage pushMessage = new MessageTemplate.PushMessage();
            MessageTemplate.PushMessage.Transmission transmission = new MessageTemplate.PushMessage.Transmission();
            transmission.setTitle(informForm.getTitle());
            transmission.setContent(informForm.getBody());
            transmission.setUrl(HtmlUtil.unescape(informForm.getUrl()));
            pushMessage.setTransmission(transmission);

            messageTemplate.setPush_message(pushMessage);
            String result = null;
            switch (informForm.getEquipmentType()) {
                //看看生活
                case "kksh":
                    String kk_sh_token_ge_tui = getAuthToken("kk_sh_token_ge_tui");
                    result = listPushToApp(device, messageTemplate, kksh.getAppId(), kk_sh_token_ge_tui);
                    break;
                //看看社保
                case "kksb":
                    String kk_sb_token_ge_tui = getAuthToken("kk_sb_token_ge_tui");
                    result = listPushToApp(device, messageTemplate, kksb.getAppId(), kk_sb_token_ge_tui);
                    break;
                case "kkwj":
                    String kk_wj_token_ge_tui = getAuthToken("kk_wj_token_ge_tui");
                    result = listPushToApp(device, messageTemplate, kkwj.getAppId(), kk_wj_token_ge_tui);
                    break;
                case "kkmz":
                    String kk_mz_token_ge_tui = getAuthToken("kk_mz_token_ge_tui");
                    result = listPushToApp(device, messageTemplate, kkmz.getAppId(), kk_mz_token_ge_tui);
                    break;
                default:
                    break;
            }
            if (result == null) {
                return CommonResult.failed();
            }
            JSONObject jsonObject = JSONObject.parseObject(result);
            if ((jsonObject.containsKey("code") && jsonObject.getInteger("code") == 0)) {
                JSONObject data = jsonObject.getJSONObject("data");
                return CommonResult.success(data);
            }
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return CommonResult.failed();
    }

    /**
     * 【toApp】执行群推
     *
     * @param messageTemplate 请求参数
     * @param token           token
     * @return 执行结果
     */
    public String listPushToApp(Integer device, MessageTemplate messageTemplate, String appId, String token) throws UnsupportedEncodingException {
        Map<String, Object> map = convertBeanToMap(device, false, 1, messageTemplate);
        //请求url
        String url = GE_TUI_BASE_URL + appId + "/push/all";
        log.info("执行群推");
        return generalPost(url, token, map);
    }

    /**
     * 实体类封装为map
     *
     * @param messageTemplate 消息模板
     * @return map对象
     */
    public Map<String, Object> convertBeanToMap(Integer device, boolean flag, int type, MessageTemplate messageTemplate) throws UnsupportedEncodingException {
        if (messageTemplate == null) {
            return null;
        }
        //设置请求唯一标识号，10-32位之间；如果request_id重复，会导致消息丢失
        Map<String, Object> map = new HashMap<>();
        map.put("request_id", String.valueOf(IdUtil.getSnowflake(1, 1).nextId()));

        //设置任务组名
        if (StringUtils.isNotBlank(messageTemplate.getGroup_name())) {
            map.put("group_name", messageTemplate.getGroup_name());
        }

        //推送条件设置
        Map<String, Object> setting = BeanUtil.beanToMap(messageTemplate.getSettings(), false, true);
        Map<String, Object> strategy = new HashMap<>();
        strategy.put("default", 4);
        strategy.put("ios", 1);
        strategy.put("st", 4);
        strategy.put("hw", 4);
        strategy.put("xm", 4);
        strategy.put("vv", 4);
        strategy.put("mz", 4);
        strategy.put("op", 4);
        setting.put("strategy", strategy);
        map.put("settings", setting);
        //获取推送目标
        if (!flag) {
            //推送目标用户该接口audience 对应值为all，表示推送所有用户
            MessageTemplate.Audience audience1 = messageTemplate.getAudience();
            if (audience1 == null) {
                map.put("audience", "all");
            } else {
                if (type == 1) {
                    Map<String, Object> audience = BeanUtil.beanToMap(audience1, false, true);
                    map.put("audience", audience);
                } else {
                    Map<String, Object> cid = new HashMap<>();
                    cid.put("cid", audience1.getAlias());
                    map.put("audience", cid);
                }
            }
        }
        //获取推送消息转为map 不转为下划线,忽略空值
        MessageTemplate.PushMessage push_message = messageTemplate.getPush_message();
        Map<String, Object> push_messages = BeanUtil.beanToMap(push_message, false, true);

        //透传消息内容，与notification、revoke 三选一，都填写时报错，长度 ≤ 3072
        MessageTemplate.PushMessage.Transmission transmission = push_message.getTransmission();
        String str = "{\"content\": \"%s\", \"push_type\": \"0\", \"title\": \"%s\", \"url\": \"%s\"}";
        String intent = String.format(str, transmission.getContent(), transmission.getTitle(), transmission.getUrl());
        push_messages.put("transmission", intent);

        //撤回消息时使用(仅撤回厂商通道消息，支持的厂商有小米、VIVO)，与notification、transmission三选一，都填写时报错(消息撤回请勿填写策略参数)
        MessageTemplate.PushMessage.Revoke revoke = push_message.getRevoke();
        if (revoke != null) {
            //设置撤回消息
            Map<String, Object> revokes = BeanUtil.beanToMap(revoke, false, true);
            push_messages.put("revoke", revokes);
        }
        //设置推送消息
        map.put("push_message", push_messages);
        //设置厂商通道
        map.put("push_channel", setChannel(push_message));
        return map;
    }

    /**
     * 设置推送渠道
     *
     * @param push_message 消息
     * @return json
     */
    public cn.hutool.json.JSONObject setChannel(MessageTemplate.PushMessage push_message) throws UnsupportedEncodingException {
        Map<String, Object> channel = new HashMap<>();
        //获取ios配置
        channel.put("ios", getIosChannel(push_message));
        //获取android配置
        channel.put("android", getAndroidChannel(push_message));
        return JSONUtil.parseFromMap(channel);
    }

    /**
     * ios厂商通道消息配置
     *
     * @param push_message 消息
     * @return ios配置
     */
    private Map<String, Object> getIosChannel(MessageTemplate.PushMessage push_message) throws UnsupportedEncodingException {
        Map<String, Object> ios = new HashMap<>();
        ios.put("type", "notify");
//        ios.put("auto_badge", "+1");
        //推送通知消息内容
        Map<String, Object> aps = new HashMap<>();
        aps.put("content-available", 0);
        //通知消息
        MessageTemplate.PushMessage.Transmission transmission = push_message.getTransmission();
        Map<String, Object> alert = new HashMap<>();
        alert.put("title", transmission.getTitle());
        alert.put("body", transmission.getContent());
        aps.put("alert", alert);
        ios.put("aps", aps);
        String str = "{\"content\": \"%s\", \"push_type\": \"0\", \"title\": \"%s\", \"url\": \"%s\"}";
        String intent = String.format(str, transmission.getContent(), transmission.getTitle(), transmission.getUrl());
        ios.put("payload", intent);
        return ios;
    }

    /**
     * 安卓厂商通道消息配置
     *
     * @param push_message 消息
     * @return android配置
     */
    private Map<String, Object> getAndroidChannel(MessageTemplate.PushMessage push_message) throws UnsupportedEncodingException {
        Map<String, Object> android = new HashMap<>();
        Map<String, Object> ups = new HashMap<>();
        MessageTemplate.PushMessage.Transmission transmission = push_message.getTransmission();
        HashMap<String, Object> notification = new HashMap<>();
        notification.put("title", transmission.getTitle());
        notification.put("body", transmission.getContent());
        notification.put("click_type", "intent");
        String pack = "com.weface.kksocialsecurity";
        String component = pack + "/com.weface.kksocialsecurity.activity.WellcomeActivity";
        String str = "intent://com.weface.kksocialsecurity/localopen?push_type=0&url=%s#Intent;scheme=kankan;package=%s;component=%s;S.kkparam={\"content\": \"%s\", \"push_type\": \"0\", \"title\": \"%s\", \"url\": \"%s\"};end";
        String intent = String.format(str, URLEncoder.encode(transmission.getUrl(), "UTF-8"), pack, component, transmission.getContent(), transmission.getTitle(), URLEncoder.encode(transmission.getUrl(), "UTF-8"));
        notification.put("intent", intent);
        ups.put("notification", notification);
        android.put("ups", ups);
        return android;
    }

    /**
     * post请求通用
     *
     * @param url   请求路径
     * @param token 鉴权
     * @param map   请求体参数
     * @return 结果
     */
    public String generalPost(String url, String token, Map<String, Object> map) {
        HttpRequest post = HttpRequest.post(url);
        //设置请求头token
        post.header("token", token);
        post.header("Content-Type", CONTENT_TYPE);
        return post.body(JSONUtil.toJsonStr(map)).execute().body();
    }
}
