package com.weface.config;

import cn.hutool.json.JSONUtil;
import com.weface.code.CommonResult;
import com.weface.code.ResultCode;
import com.weface.common.utils.CommonUtil;
import com.weface.common.utils.Constant;
import com.weface.common.utils.IPUtils;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.FilterConfig;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * @author Aleyn
 * @since 2021/11/20 13:00
 */
@Slf4j
public class SignValidateFilter implements Filter {

    /**
     * 参数签名过期时间  2分钟
     */
    public static final long EXPIRE_TIME = 1000 * 60 * 2;
    public static final String TIME_NAME = "timestamp";
    public static final String SIGN_NAME = "sign";
    public static final String SECRET_KEY = "KkweFace95271125";
    /**
     * 排除链接
     */
    public List<String> excludes = new ArrayList<>();

    @Override
    public void init(FilterConfig filterConfig) {
        log.info("初始化filter");
        //不拦截的部分
        String tempExcludes = filterConfig.getInitParameter("excludes");
        if (!CommonUtil.isEmpty(tempExcludes)) {
            String[] url = tempExcludes.split(",");
            excludes.addAll(Arrays.asList(url));
        }
    }


    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        corsConfig(response);
        String ipAddr = IPUtils.getIpAddr(request);
        log.info("调用了{}", request.getServletPath() + "方法");
        log.info("请求IP地址为:{}", ipAddr);

        //不拦截的
        if (handleExcludeUrl(request)) {
            filterChain.doFilter(servletRequest, servletResponse);
        } else {
            //进行签名校验
            checkParamAndSign(request, response, filterChain);
        }

    }

    /**
     * 设置允许跨域
     *
     * @param response 响应头
     */
    private void corsConfig(HttpServletResponse response) {
        response.setHeader("Access-Control-Allow-Credentials", "true");
        response.setHeader("Access-Control-Allow-Origin", "*");
        response.setHeader("Access-Control-Allow-Credentials", "true");
        response.setHeader("Access-Control-Allow-Methods", "*");
        //这里“Access-Token”是我要传到后台的内容key
        response.setHeader("Access-Control-Allow-Headers", "*");
        response.setHeader("Access-Control-Expose-Headers", "*");
    }

    /**
     * 签名校验
     */
    public void checkParamAndSign(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws IOException, ServletException {
        String timestamp = request.getHeader(TIME_NAME);
        String sign = request.getHeader(SIGN_NAME);
        if (CommonUtil.isEmpty(timestamp) || CommonUtil.isEmpty(sign)) {
            this.fail(response, ResultCode.NOT_SIGN_ERROR);
            return;
        }
        long now = System.currentTimeMillis();
        if (now - (Long.parseLong(timestamp)) > EXPIRE_TIME) {
            this.fail(response, ResultCode.REQUEST_EXPIRE_ERROR);
            return;
        }
        Map<String, Object> param = new HashMap<>();
        param.put(TIME_NAME, timestamp);
        //请求类型
        String contentType = request.getContentType() == null ? "" : request.getContentType().trim().toLowerCase();
        //json参数
        boolean isJson = contentType.contains("application/json");
        //iso中使用application/x-www-form-urlencoded请求时，参数会转为multipart/form-data模式
        boolean isMultiPart = contentType.contains("multipart/form-data");
        OnceRequestWrapper requestWrapper = null;
        if (isJson) {
            requestWrapper = new OnceRequestWrapper(request);
            //获取参数
            String body = requestWrapper.getBody();
            if (!CommonUtil.isEmpty(body)) {
                if (JSONUtil.isJson(body)) {
                    Map<String, Object> bodyMap = JSONUtil.toBean(body, Map.class);
                    param.putAll(bodyMap);
                } else {
                    //请求参数不是json
                    fail(response, ResultCode.CANNOT_ANALYSIS_PARAM_ERROR);
                }
            }
        } else if (isMultiPart) {
            String data = request.getParameter("Data");
            if (data != null && JSONUtil.isJson(data)) {
                //参数转为map
                Map map = JSONUtil.parseObj(data).toBean(Map.class);
                param.putAll(map);
                requestWrapper = new OnceRequestWrapper(request, param);
            }
        } else {
            //其他参数
            Enumeration<String> names = request.getParameterNames();
            Map<String, Object> parameterMap = new HashMap<>();
            while (names.hasMoreElements()) {
                String element = names.nextElement();
                String value = request.getParameter(element);
                parameterMap.put(element, value);
            }
            param.putAll(parameterMap);
        }

        if (CommonUtil.apiParamSign(param, SECRET_KEY).equals(sign)) {
            if (isJson | isMultiPart) {
                filterChain.doFilter(requestWrapper, response);
            } else {
                filterChain.doFilter(request, response);
            }
        } else {
            fail(response, ResultCode.SIGN_VALID_FAIL_ERROR);
        }

    }


    private boolean handleExcludeUrl(HttpServletRequest request) {
        String url = request.getServletPath();
        String method = request.getMethod();
        // GET DELETE 不过滤
        if (method == null || method.matches(Constant.MethodType.GET.getValue()) || method.matches(Constant.MethodType.DELETE.getValue())) {
            return true;
        }
        return CommonUtil.matches(url, excludes);
    }

    @Override
    public void destroy() {
        log.info("销毁filter");
    }

    public void fail(ServletResponse response, ResultCode code) {
        response.setContentType("application/json;charset=UTF-8");
        CommonResult<Object> failed = CommonResult.failed(code);
        try {
            ServletOutputStream outputStream = response.getOutputStream();
            outputStream.write(JSONUtil.toJsonStr(failed).getBytes(StandardCharsets.UTF_8));
        } catch (IOException e) {
            log.error(e.getMessage());
        }
    }
}
