package com.weface.common.exception;

import com.weface.common.utils.Model;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.http.converter.HttpMessageConversionException;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolationException;

/**
 * 异常处理器
 *
 * @author Mark sunlightcs@gmail.com
 */
@RestControllerAdvice
public class RRExceptionHandler {
    private Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * 处理自定义异常
     */
    @ExceptionHandler(RRException.class)
    public Model handleRRException(RRException e) {
        return Model.error(e.getCode(), e.getMsg());
    }

    @ExceptionHandler(NoHandlerFoundException.class)
    public Model handlerNoFoundException(Exception e) {
        logger.error(e.getMessage(), e);
        return Model.error(404, "路径不存在，请检查路径是否正确");
    }

    @ExceptionHandler(DuplicateKeyException.class)
    public Model handleDuplicateKeyException(DuplicateKeyException e) {
        logger.error(e.getMessage(), e);
        return Model.error("数据库中已存在该记录");
    }


    @ExceptionHandler(Exception.class)
    public Model handleException(Exception e) {
        logger.error(e.getMessage(), e);
        return Model.error();
    }

    /**
     * 缺失请求参数处理
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(MissingServletRequestParameterException.class)
    public Model handleMissingServletRequestParameterException(MissingServletRequestParameterException e, HttpServletRequest request) {
        String message = "缺失请求参数" + e.getParameterName();
        logger.error(e.getMessage());
        return Model.error(400, message);
    }

    /**
     * 缺失请求参数处理
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(ConstraintViolationException.class)
    public Model handleConstraintViolationException(ConstraintViolationException e, HttpServletRequest request) {
        String message = "缺失请求参数" + e.getMessage();
        logger.error(e.getMessage());
        return Model.error(400, message);
    }



    /**
     * 请求参数类型错误处理
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public Model handleMethodArgumentTypeMismatchException(MethodArgumentTypeMismatchException e, HttpServletRequest request) {
        String message = "请求参数" + e.getName() + "类型错误";
        logger.error(e.getMessage());
        return Model.error(400, message);
    }


    /**
     * 请求Conten-Type 错误
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public Model handleHttpMediaTypeNotSupportedException(HttpMediaTypeNotSupportedException e, HttpServletRequest request) {
        logger.error(e.getMessage());
        return Model.error(400, "请求Conten-Type 错误");
    }

    /**
     * 参数类型错误异常类型处理
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(HttpMessageConversionException.class)
    public Model handleHttpMessageNotReadableException(HttpMessageConversionException e, HttpServletRequest request) {
        String message = "参数类型错误";
        logger.error(e.getMessage());
        return Model.error(400, message);
    }


    /**
     * 空指针异常处理
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(NullPointerException.class)
    public Model handleNullPointerException(NullPointerException e, HttpServletRequest request) {
        String message = "系统出错了...";
        logger.error(e.getMessage());
        return Model.error(500, message);
    }


    /**
     * 绑定异常处理
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler(BindException.class)
    public Model handleBindException(BindException e, HttpServletRequest request) {
        BindingResult result = e.getBindingResult();
        StringBuilder errorMsg = new StringBuilder();
        for (ObjectError error : result.getAllErrors()) {
            errorMsg.append(error.getDefaultMessage()).append(",");
        }
        errorMsg.delete(errorMsg.length() - 1, errorMsg.length());
        return Model.error(400, "参数错误" + errorMsg);
    }


    /**
     * 处理运行时异常
     *
     * @param e
     * @param request
     * @return
     */
    @ExceptionHandler({RuntimeException.class})
    @ResponseBody
    public Model handleRuntimeException(RuntimeException e, HttpServletRequest request) {
        logger.error(e.getMessage());
        return Model.error(400, "");
    }


}
