package com.weface.common.validator;

import com.weface.code.CommonResult;
import com.weface.code.ResultCode;
import com.weface.dto.InformForm;
import com.weface.dto.MultiInformForm;
import com.weface.dto.MultiMsgDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author : Administrator
 * @date : 2022/4/21 14:28
 */
@Slf4j
public class ValidatorParam {

    /**
     * 校验推送参数
     *
     * @param informForm 推送参数
     * @return 核验结果
     */
    public static CommonResult validPushParams(InformForm informForm) {
        Integer style = informForm.getStyle();
        if (Objects.nonNull(style)) {
            if (style == 1 || style == 2) {
                String content = informForm.getContent();
                if (StringUtils.isEmpty(content)) {
                    return CommonResult.failed(ResultCode.PARAMS_ERROR);
                }
            }
        }
        return null;
    }

    /**
     * 核验别名单推推送参数
     *
     * @param informForm 推送参数
     * @return 核验结果
     */
    public static CommonResult validPushSingleAlias(InformForm informForm) {
        String body = informForm.getBody();
        if (body.length() > 60) {
            return CommonResult.failed(ResultCode.CONTENT_MORE_LENGTH);
        }
        Integer pushType = informForm.getPushType();
        if (pushType != null && pushType == 1) {
            String className = informForm.getClassName();
            String classTitle = informForm.getClassTitle();
            String needLogin = informForm.getNeedLogin();
            if (StringUtils.isEmpty(className) || StringUtils.isEmpty(classTitle) || StringUtils.isEmpty(needLogin)) {
                return CommonResult.failed(ResultCode.PARAMS_ERROR);
            }
        }
        String cid = informForm.getCid();
        String phone = informForm.getPhone();
        Integer messageTemplate = informForm.getMessageTemplate();
        if (StringUtils.isEmpty(cid) || StringUtils.isEmpty(phone) || messageTemplate == null) {
            return CommonResult.failed(ResultCode.PARAMS_ERROR);
        }
        return null;
    }


    /**
     * 校验推送参数
     *
     * @param informForm 推送参数
     * @return 核验结果
     */
    public static CommonResult validBatchPushParams(MultiInformForm informForm) {
        List<MultiMsgDTO> msgList = informForm.getMsgList();
        List<MultiMsgDTO> blankTitleList = msgList.stream().filter(x -> StringUtils.isBlank(x.getTitle())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(blankTitleList)) {
            log.error("msgList中含title为空的消息: param=[{}]", informForm);
            return CommonResult.validateFailed("msgList中含title为空的消息");
        }

        List<MultiMsgDTO> blankContentList = msgList.stream().filter(x -> StringUtils.isBlank(x.getContent())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(blankContentList)) {
            log.error("msgList中含content为空的消息: param=[{}]", informForm);
            return CommonResult.validateFailed("msgList中含content为空的消息");
        }

        List<MultiMsgDTO> emptyAliasList = msgList.stream().filter(x -> x.getAlias() == null || x.getAlias().length == 0).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(emptyAliasList)) {
            log.error("msgList中含alias为空的消息: param=[{}]", informForm);
            return CommonResult.validateFailed("msgList中含alias为空的消息");
        }

        return null;
    }

    /**
     * 根据地理位置推送校验
     * @param informForm
     * @return
     */
    public static CommonResult validRegion(InformForm informForm) {
        if (StringUtils.isBlank(informForm.getCity()) && StringUtils.isBlank(informForm.getProvince()) && StringUtils.isBlank(informForm.getCountry())) {
            log.error("region中至少有一个字段不为空", informForm);
            return CommonResult.validateFailed("region中至少有一个字段不为空");
        }

        return null;
    }

}
