package com.weface.common.utils;

import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import org.locationtech.spatial4j.context.SpatialContext;
import org.locationtech.spatial4j.distance.DistanceUtils;
import org.locationtech.spatial4j.shape.Rectangle;
import org.springframework.util.AntPathMatcher;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.util.*;
import java.util.function.Function;
import java.util.regex.Pattern;

/**
 * @author Administrator
 */
public class CommonUtil {

    private static final SpatialContext SPATIAL_CONTEXT = SpatialContext.GEO;

    /**
     * 判断字符串是否Base64字符串
     */
    public static boolean isBase64(String str) {
        if (str == null || "".equals(str)) {
            return false;
        }
        String reg = "^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{4}|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)$";
        return Pattern.matches(reg, str);
    }

    /**
     * 将一个集合平均分成n组
     */
    public static <T> List<List<T>> averageAssign(List<T> source, int n) {
        List<List<T>> result = new ArrayList<>();
        //(先计算出余数)
        int remainder = source.size() % n;
        //然后是商
        int number = source.size() / n;
        //偏移量
        int offset = 0;
        for (int i = 0; i < n; i++) {
            List<T> value;
            if (remainder > 0) {
                value = source.subList(i * number + offset, (i + 1) * number + offset + 1);
                remainder--;
                offset++;
            } else {
                value = source.subList(i * number + offset, (i + 1) * number + offset);
            }
            result.add(value);
        }
        return result;
    }

    /**
     * 将一个集合分成若干组，每组n个元素
     */
    public static <T> List<List<T>> fixedGrouping(List<T> source, int n) {

        if (null == source || source.size() == 0 || n <= 0) {
            return null;
        }
        List<List<T>> result = new ArrayList<>();
        int remainder = source.size() % n;
        int size = (source.size() / n);
        for (int i = 0; i < size; i++) {
            List<T> subset;
            subset = source.subList(i * n, (i + 1) * n);
            result.add(subset);
        }
        if (remainder > 0) {
            List<T> subset = source.subList(size * n, size * n + remainder);
            result.add(subset);
        }
        return result;
    }


    /**
     * 对实体类的某些字段进行加密
     *
     * @param t          实体类
     * @param encryptFun 加密的方法
     * @param fieldNames 要加密的字段数组
     * @param <T>        TT
     * @return T
     */
    public static <T> T encryptEntity(T t, Function<String, String> encryptFun, String[] fieldNames) {
        for (String name : fieldNames) {
            Class<?> clazz = t.getClass();
            try {
                Field field = clazz.getDeclaredField(name);
                Class<?> fileType = field.getType();
                //加密的字段，数据类型 必须是String
                if (fileType == String.class) {
                    //设置字段为可操作
                    field.setAccessible(true);
                    String value = (String) field.get(t);
                    if (value == null) {
                        continue;
                    }
                    String encryptValue = encryptFun.apply(value);
                    //赋值为加密后的字段
                    field.set(t, encryptValue);
                    field.setAccessible(false);
                }
            } catch (NoSuchFieldException | IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return t;
    }


    /**
     * 对实体类的某些字段进行解码
     *
     * @param t          实体类对象
     * @param decryptFun 解码的方法
     * @param fieldNames 解码的字段数组
     * @param <T>        TT
     * @return T
     */
    public static <T> T decryptEntity(T t, Function<String, String> decryptFun, String[] fieldNames) {
        for (String name : fieldNames) {
            Class<?> clazz = t.getClass();
            try {
                Field field = clazz.getDeclaredField(name);
                Class<?> fileType = field.getType();
                //加密的字段，数据类型 必须是String
                if (fileType == String.class) {
                    //设置字段为可操作
                    field.setAccessible(true);
                    String value = (String) field.get(t);
                    //如果是base64，(加密后的数据时base字符串)，再进行解码
                    if (value != null && isBase64(value)) {
                        String decryptStr = decryptFun.apply(value);
                        //赋值为加密后的字段
                        field.set(t, decryptStr);
                    }
                    field.setAccessible(false);
                }
            } catch (NoSuchFieldException | IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return t;
    }

    /**
     * 签名的规则：sk + 所有非空参数值拼接 + sk
     *
     * @param param 所有请求参数
     * @return sign 签名
     */
    public static String apiParamSign(Map<String, Object> param, String sk) {
        TreeMap<String, Object> map = new TreeMap<>(param);
        StringBuilder builder = new StringBuilder();
        builder.append(sk);
        map.forEach((k, v) -> {
            if (v != null && !"null".equals(v) && !"".equals(v)) {
                builder.append(v);
            }
        });
        builder.append(sk);
        return SecureUtil.md5(builder.toString());
    }

    /**
     * 调用百度接口，获取地址信息的经纬度
     *
     * @param address 地址信息
     * @return double[] 0经度 1纬度
     */
    public static double[] getAxisByAddress(String address) {
        String url = "http://api.map.baidu.com/geocoding/v3/?address=%s&ret_coordtype=wgs84ll&output=json&ak=b9kvzxPGvOxvcPllM9P7s81C85bKQPgE";
        String request = String.format(url, address);
        JSONObject jsonObject = JSONUtil.parseObj(HttpUtil.get(request));
        if (jsonObject.getInt(Constant.STATUS) == 0) {
            JSONObject location = jsonObject.getJSONObject("result").getJSONObject("location");
            Double lng = location.getDouble("lng");
            Double lat = location.getDouble("lat");
            return new double[]{lng, lat};
        }
        return null;
    }

    /**
     * 利用开源库计算外接正方形坐标
     *
     * @param distance 距离，单位千米
     * @param userLng  当前经度
     * @param userLat  当前纬度
     * @return 返回正方形坐标
     */
    public static Rectangle getRectangle(double distance, double userLng, double userLat) {
        return SPATIAL_CONTEXT.getDistCalc()
                .calcBoxByDistFromPt(SPATIAL_CONTEXT.getShapeFactory().pointXY(userLng, userLat),
                        distance * DistanceUtils.KM_TO_DEG, SPATIAL_CONTEXT, null);
    }


    /***
     * 球面中，两点间的距离（第三方库方法）
     *
     * @param longitude 经度1
     * @param latitude  纬度1
     * @param userLng   经度2
     * @param userLat   纬度2
     * @return 返回距离，单位km
     */
    public static double getDistance(Double longitude, Double latitude, double userLng, double userLat) {
        return SPATIAL_CONTEXT.calcDistance(SPATIAL_CONTEXT.getShapeFactory().pointXY(userLng, userLat),
                SPATIAL_CONTEXT.getShapeFactory().pointXY(longitude, latitude)) * DistanceUtils.DEG_TO_KM;
    }


    /**
     * * 判断一个字符串是否为空串
     *
     * @param str String
     * @return true：为空 false：非空
     */
    public static boolean isEmpty(String str) {
        return str == null || str.length() == 0;
    }

    /**
     * * 判断一个Collection是否为空， 包含List，Set，Queue
     *
     * @param coll 要判断的Collection
     * @return true：为空 false：非空
     */
    public static boolean isEmpty(Collection<?> coll) {
        return coll == null || coll.isEmpty();
    }

    /**
     * 查找指定字符串是否匹配指定字符串列表中的任意一个字符串
     *
     * @param str  指定字符串
     * @param strs 需要检查的字符串数组
     * @return 是否匹配
     */
    public static boolean matches(String str, List<String> strs) {
        if (isEmpty(str) || isEmpty(strs)) {
            return false;
        }
        for (String pattern : strs) {
            if (isMatch(pattern, str)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断url是否与规则配置:
     * ? 表示单个字符;
     * * 表示一层路径内的任意字符串，不可跨层级;
     * ** 表示任意层路径;
     *
     * @param pattern 匹配规则
     * @param url     需要匹配的url
     * @return boolean 是否匹配
     */
    public static boolean isMatch(String pattern, String url) {
        AntPathMatcher matcher = new AntPathMatcher();
        return matcher.match(pattern, url);
    }

    /**
     * 获取真实ip地址，避免获取代理ip
     */
    public static String getIpAddress(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || Constant.UN_KNOW.equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Real-IP");
        }
        if (ip == null || ip.length() == 0 || Constant.UN_KNOW.equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || Constant.UN_KNOW.equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || Constant.UN_KNOW.equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");
        }
        if (ip == null || ip.length() == 0 || Constant.UN_KNOW.equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_X_FORWARDED_FOR");
        }

        if (ip == null || ip.length() == 0 || Constant.UN_KNOW.equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        return ip;
    }


    /**
     * 加密ID和手机号
     *
     * @param uid uid
     * @param tel 手机号
     * @return 密文
     */
    public static String encryptIdAndTel(String uid, String tel) {
        //kkv+id+ 手机号码后五位+手机号码第六位+手机号码前五位
        try {
            String prefix = "kkv";
            String sub1 = StrUtil.sub(tel, 0, 5);
            String sub2 = StrUtil.sub(tel, 5, 6);
            String sub3 = StrUtil.sub(tel, 6, 11);
            return prefix + uid + sub3 + sub2 + sub1;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 解析ID和手机号
     *
     * @param encryptStr 密文
     * @return 明文
     */
    public static String[] decryptIdAndTel(String encryptStr) {
        //kkv+id+ 手机号码后五位+手机号码第六位+手机号码前五位
        try {
            String prefix = "kkv";
            int length = encryptStr.length();
            String hexTel = encryptStr.substring(length - 11, length);
            String sub1 = StrUtil.sub(hexTel, 0, 5);
            String sub2 = StrUtil.sub(hexTel, 5, 6);
            String sub3 = StrUtil.sub(hexTel, 6, 11);
            String tel = sub3 + sub2 + sub1;
            String suffix = StrUtil.subAfter(encryptStr, prefix, true);
            String uid = StrUtil.sub(suffix, 0, suffix.length() - 11);
            String[] arr = new String[2];
            arr[0] = uid;
            arr[1] = tel;
            return arr;
        } catch (Exception e) {
            return null;
        }
    }
}
